
import { SectionWrapper } from "@/components/SectionWrapper";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import type { TechStackCategory, TechStackItem } from "@/lib/data";
import { Skeleton } from "@/components/ui/skeleton";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";
import { AlertCircle, Code } from "lucide-react";

interface TechStackSectionProps {
  techStackCategories: TechStackCategory[] | null;
  isLoading: boolean;
  error: string | null;
}

export function TechStackSection({ techStackCategories, isLoading, error }: TechStackSectionProps) {
  if (isLoading) {
    return (
      <SectionWrapper id="stack" title="Key Skills" subtitle="Core competencies and technologies I leverage.">
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-2 gap-8">
          {[...Array(2)].map((_, i) => (
            <Card key={i} className="shadow-lg">
              <CardHeader>
                <Skeleton className="h-7 w-1/2 mb-2" />
              </CardHeader>
              <CardContent className="flex flex-wrap gap-3">
                {[...Array(4)].map((_, j) => (
                  <Skeleton key={j} className="h-8 w-24 rounded-full" />
                ))}
              </CardContent>
            </Card>
          ))}
        </div>
      </SectionWrapper>
    );
  }

  if (error) {
    return (
      <SectionWrapper id="stack" title="Key Skills" subtitle="Core competencies and technologies I leverage.">
        <Alert variant="destructive" className="max-w-2xl mx-auto">
          <AlertCircle className="h-4 w-4" />
          <AlertTitle>Could Not Load Skills</AlertTitle>
          <AlertDescription>{error}</AlertDescription>
        </Alert>
      </SectionWrapper>
    );
  }

  if (!techStackCategories || techStackCategories.length === 0) {
    return (
      <SectionWrapper id="stack" title="Key Skills" subtitle="Core competencies and technologies I leverage.">
        <Alert variant="default" className="max-w-2xl mx-auto">
          <AlertCircle className="h-4 w-4" />
          <AlertTitle>No Skills Data Available</AlertTitle>
          <AlertDescription>Skills information is currently not available.</AlertDescription>
        </Alert>
      </SectionWrapper>
    );
  }

  return (
    <SectionWrapper
      id="stack"
      title="Key Skills"
      subtitle="Core competencies and technologies I leverage."
    >
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-2 gap-8">
        {techStackCategories.map((category) => (
          <Card
            key={category.category}
            className="shadow-xl hover:shadow-2xl transition-shadow duration-300 bg-card text-card-foreground"
          >
            <CardHeader>
              <CardTitle className="text-2xl text-primary">{category.category}</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="flex flex-wrap gap-3">
                {category.items.map((tech: TechStackItem) => {
                  const IconComponent = tech.icon || Code; // Fallback to Code icon
                  return (
                    <Badge
                      key={tech.name}
                      variant="secondary" // Base variant, overridden by className for specific bg/text
                      className="text-sm px-4 py-2 rounded-full flex items-center gap-2 bg-accent text-accent-foreground hover:bg-accent/90 cursor-default"
                    >
                      <span className="bg-clip-text text-transparent bg-gradient-to-r from-[hsl(var(--accent-foreground))] via-[hsl(var(--shimmer-highlight))] to-[hsl(var(--accent-foreground))] bg-[size:300%_300%] animate-gradient-xy inline-flex items-center justify-center">
                        <IconComponent className="h-5 w-5" />
                      </span>
                      {tech.name}
                    </Badge>
                  );
                })}
              </div>
            </CardContent>
          </Card>
        ))}
      </div>
    </SectionWrapper>
  );
}
