
"use client";

import { SectionWrapper } from "@/components/SectionWrapper";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import type { Interest } from "@/lib/data";
import { Skeleton } from "@/components/ui/skeleton";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";
import { AlertCircle, Smile } from "lucide-react";

interface InterestsSectionProps {
  interests: Interest[] | null;
  isLoading: boolean;
  error: string | null;
}

export function InterestsSection({ interests, isLoading, error }: InterestsSectionProps) {
  if (isLoading) {
    return (
      <SectionWrapper id="interests" title="Personal Interests" subtitle="Beyond the code and project plans, here's what I enjoy.">
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {[...Array(3)].map((_, i) => (
            <Card key={i} className="shadow-lg">
              <CardHeader className="flex flex-row items-center gap-4 p-4">
                <Skeleton className="h-10 w-10 rounded-lg" />
                <Skeleton className="h-6 w-3/4" />
              </CardHeader>
              <CardContent className="pt-0 p-4">
                <Skeleton className="h-3 w-full" />
                <Skeleton className="h-3 w-5/6 mt-1" />
              </CardContent>
            </Card>
          ))}
        </div>
      </SectionWrapper>
    );
  }
  
  if (error) {
    return (
      <SectionWrapper id="interests" title="Personal Interests" subtitle="Beyond the code and project plans, here's what I enjoy.">
        <Alert variant="destructive" className="max-w-2xl mx-auto">
          <AlertCircle className="h-4 w-4" />
          <AlertTitle>Could Not Load Interests</AlertTitle>
          <AlertDescription>{error}</AlertDescription>
        </Alert>
      </SectionWrapper>
    );
  }

  if (!interests || interests.length === 0) {
     return (
      <SectionWrapper id="interests" title="Personal Interests" subtitle="Beyond the code and project plans, here's what I enjoy.">
        <Alert variant="default" className="max-w-2xl mx-auto">
          <AlertCircle className="h-4 w-4" />
          <AlertTitle>No Interests Data Available</AlertTitle>
          <AlertDescription>Personal interests information is currently not available.</AlertDescription>
        </Alert>
      </SectionWrapper>
    );
  }

  return (
    <SectionWrapper
      id="interests"
      title="Personal Interests"
      subtitle="Beyond the code and project plans, here's what I enjoy."
    >
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        {interests.map((interest: Interest) => {
          const IconComponent = interest.icon || Smile;
          return (
            <Card 
              key={interest.name} 
              className="shadow-lg hover:shadow-xl transition-shadow duration-300 bg-card text-card-foreground flex flex-col"
            >
              <CardHeader className="flex flex-row items-center gap-4 p-4">
                <div className="flex-shrink-0 bg-primary text-primary-foreground p-3 rounded-lg">
                  <span className="bg-clip-text text-transparent bg-gradient-to-r from-[hsl(var(--primary-foreground))] via-[hsl(var(--shimmer-highlight))] to-[hsl(var(--accent))] bg-[size:300%_300%] animate-gradient-xy inline-flex items-center justify-center">
                    <IconComponent className="h-6 w-6" />
                  </span>
                </div>
                <CardTitle className="text-lg font-semibold text-primary">{interest.name}</CardTitle>
              </CardHeader>
              {interest.description && (
                <CardContent className="pt-0 p-4 flex-grow">
                  <CardDescription className="text-sm text-muted-foreground">
                    {interest.description}
                  </CardDescription>
                </CardContent>
              )}
            </Card>
          );
        })}
      </div>
    </SectionWrapper>
  );
}
