
import { SectionWrapper } from "@/components/SectionWrapper";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import type { ExperienceEntry } from "@/lib/data";
import { Skeleton } from "@/components/ui/skeleton";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";
import { AlertCircle, Briefcase } from "lucide-react";

interface ExperienceSectionProps {
  experienceEntries: ExperienceEntry[] | null;
  isLoading: boolean;
  error: string | null;
}

export function ExperienceSection({ experienceEntries, isLoading, error }: ExperienceSectionProps) {
  if (isLoading) {
    return (
      <SectionWrapper 
        id="experience" 
        title="Professional Experience" 
        subtitle="My journey through the tech world, one role at a time."
      >
        <div className="relative space-y-12">
          {[...Array(2)].map((_, i) => (
            <div key={i} className="relative pl-10">
              <Skeleton className="absolute left-0 top-1 h-10 w-10 rounded-full" />
              <Card className="ml-4 shadow-lg">
                <CardHeader>
                  <Skeleton className="h-5 w-3/4 mb-1" />
                  <Skeleton className="h-4 w-1/2" />
                </CardHeader>
                <CardContent>
                  <Skeleton className="h-3 w-full mt-2" />
                  <Skeleton className="h-3 w-5/6 mt-1" />
                  <Skeleton className="h-3 w-full mt-1" />
                </CardContent>
              </Card>
            </div>
          ))}
        </div>
      </SectionWrapper>
    );
  }

  if (error) {
    return (
      <SectionWrapper 
        id="experience" 
        title="Professional Experience" 
        subtitle="My journey through the tech world, one role at a time."
      >
        <Alert variant="destructive" className="max-w-2xl mx-auto">
          <AlertCircle className="h-4 w-4" />
          <AlertTitle>Could Not Load Experience</AlertTitle>
          <AlertDescription>{error}</AlertDescription>
        </Alert>
      </SectionWrapper>
    );
  }

  if (!experienceEntries || experienceEntries.length === 0) {
    return (
      <SectionWrapper 
        id="experience" 
        title="Professional Experience" 
        subtitle="My journey through the tech world, one role at a time."
      >
        <Alert variant="default" className="max-w-2xl mx-auto">
          <AlertCircle className="h-4 w-4" />
          <AlertTitle>No Experience Data Available</AlertTitle>
          <AlertDescription>Professional experience information is currently not available.</AlertDescription>
        </Alert>
      </SectionWrapper>
    );
  }

  return (
    <SectionWrapper 
      id="experience" 
      title="Professional Experience" 
      subtitle="My journey through the tech world, one role at a time."
    >
      <div className="relative space-y-12 before:absolute before:inset-0 before:ml-5 before:h-full before:w-0.5 before:bg-gradient-to-b before:from-primary/50 before:via-primary/30 before:to-transparent dark:before:from-primary/30 dark:before:via-primary/20">
        {experienceEntries.map((job, index) => {
          const IconComponent = job.icon || Briefcase;
          return (
            <div key={index} className="relative pl-10">
              <div className="absolute left-0 top-1 flex h-10 w-10 items-center justify-center rounded-full bg-primary text-primary-foreground shadow-md">
                 <span className="bg-clip-text text-transparent bg-gradient-to-r from-[hsl(var(--primary-foreground))] via-[hsl(var(--shimmer-highlight))] to-[hsl(var(--accent))] bg-[size:300%_300%] animate-gradient-xy inline-flex items-center justify-center">
                  <IconComponent className="h-5 w-5" />
                </span>
              </div>
              <Card 
                className="ml-4 shadow-xl hover:shadow-2xl transition-shadow duration-300 bg-card text-card-foreground"
              >
                <CardHeader>
                  <CardTitle className="text-xl font-semibold">{job.role}</CardTitle>
                  <CardDescription className="text-md text-primary">{job.company} <span className="text-sm text-muted-foreground">({job.dates})</span></CardDescription>
                </CardHeader>
                <CardContent>
                  <ul className="list-disc list-inside space-y-2 text-card-foreground/80">
                    {job.description.map((item, i) => (
                      <li key={i}>{item}</li>
                    ))}
                  </ul>
                </CardContent>
              </Card>
            </div>
          );
        })}
      </div>
    </SectionWrapper>
  );
}
