
import type { EducationEntry } from "@/lib/data";
import { SectionWrapper } from "@/components/SectionWrapper";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Skeleton } from "@/components/ui/skeleton";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";
import { AlertCircle, GraduationCap } from "lucide-react";

interface EducationSectionProps {
  educationItems: EducationEntry[] | null;
  isLoading: boolean;
  error: string | null;
}

export function EducationSection({ educationItems, isLoading, error }: EducationSectionProps) {
  if (isLoading) {
    return (
      <SectionWrapper
        id="education"
        title="Education & Certifications"
        subtitle="My academic background, professional qualifications, and relevant courses."
      >
        <div className="space-y-6">
          {[...Array(2)].map((_, i) => (
            <Card key={i} className="shadow-lg">
              <CardHeader className="flex flex-row items-center gap-4 p-4">
                <Skeleton className="h-12 w-12 rounded-full" />
                <div className="space-y-2 flex-grow">
                  <Skeleton className="h-4 w-3/4" />
                  <Skeleton className="h-3 w-1/2" />
                  <Skeleton className="h-3 w-1/4" />
                </div>
              </CardHeader>
              <CardContent className="pt-0 p-4">
                <Skeleton className="h-3 w-full" />
                <Skeleton className="h-3 w-5/6 mt-2" />
              </CardContent>
            </Card>
          ))}
        </div>
      </SectionWrapper>
    );
  }

  if (error) {
    return (
      <SectionWrapper
        id="education"
        title="Education & Certifications"
        subtitle="My academic background, professional qualifications, and relevant courses."
      >
        <Alert variant="destructive" className="max-w-2xl mx-auto">
          <AlertCircle className="h-4 w-4" />
          <AlertTitle>Could Not Load Education Details</AlertTitle>
          <AlertDescription>
            {error} Please try again later.
          </AlertDescription>
        </Alert>
      </SectionWrapper>
    );
  }

  if (!educationItems || educationItems.length === 0) {
    return (
      <SectionWrapper
        id="education"
        title="Education & Certifications"
        subtitle="My academic background, professional qualifications, and relevant courses."
      >
        <Alert variant="default" className="max-w-2xl mx-auto">
          <AlertCircle className="h-4 w-4" />
          <AlertTitle>No Education Details Available</AlertTitle>
          <AlertDescription>
            Education and certification information is currently not available.
          </AlertDescription>
        </Alert>
      </SectionWrapper>
    );
  }

  return (
    <SectionWrapper
      id="education"
      title="Education & Certifications"
      subtitle="My academic background, professional qualifications, and relevant courses."
    >
      <div className="space-y-6">
        {educationItems.map((edu, index) => {
          const IconComponent = edu.icon || GraduationCap;
          return (
            <Card
              key={edu.id || index}
              className="shadow-xl hover:shadow-2xl transition-shadow duration-300 bg-card text-card-foreground"
            >
              <CardHeader className="flex flex-row items-center gap-4 p-4">
                 <div className="flex-shrink-0 bg-primary text-primary-foreground p-3 rounded-full">
                   <span className="bg-clip-text text-transparent bg-gradient-to-r from-[hsl(var(--primary))] via-[hsl(var(--shimmer-highlight))] to-[hsl(var(--accent))] bg-[size:300%_300%] animate-gradient-xy inline-flex items-center justify-center">
                      <IconComponent className="h-6 w-6" />
                  </span>
                </div>
                <div>
                  <CardTitle className="text-lg font-semibold">{edu.item}</CardTitle>
                  <p className="text-sm text-muted-foreground">{edu.institution}</p>
                  <p className="text-xs text-muted-foreground/80">{edu.dates}</p>
                </div>
              </CardHeader>
              {(edu.description || edu.fieldOfStudy || edu.gradeOrScore) && (
                <CardContent className="pt-0 p-4 text-sm text-card-foreground/80">
                  {edu.fieldOfStudy && <p><strong>Field of Study:</strong> {edu.fieldOfStudy}</p>}
                  {edu.gradeOrScore && <p><strong>Grade/Score:</strong> {edu.gradeOrScore}</p>}
                  {edu.description && <p className="mt-2">{edu.description}</p>}
                </CardContent>
              )}
            </Card>
          );
        })}
      </div>
    </SectionWrapper>
  );
}
