
"use client";

import Link from "next/link";
import { useState } from "react";
import { motion } from "framer-motion";
import { Menu, X, Download, Briefcase, Newspaper, LogIn, UserPlus } from "lucide-react"; // Added LogIn, UserPlus
import { Button } from "@/components/ui/button";
import { ThemeToggle } from "./ThemeToggle";
import { personalInfo, type NavigationLink, defaultNavigationLinks, getIcon } from "@/lib/data";
import { usePathname } from "next/navigation";
import { Skeleton } from "../ui/skeleton";
import { Alert, AlertDescription, AlertTitle } from "../ui/alert";
import { AlertCircle } from "lucide-react";

interface HeaderProps {
  onDownloadResume?: () => void;
  navigationLinks: NavigationLink[] | null;
  isLoading?: boolean;
  error?: string | null;
}

export function Header({ onDownloadResume, navigationLinks: dynamicLinks, isLoading, error }: HeaderProps) {
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);
  const pathname = usePathname();

  // Determine which links to use: API-fetched or default, plus auth links
  const getEffectiveNavLinks = () => {
    let baseLinks = (Array.isArray(dynamicLinks) && dynamicLinks.length > 0) ? dynamicLinks : defaultNavigationLinks;
    
    // TODO: Add logic here to show "Logout" / "Profile" if user is authenticated
    // For now, always show Login/Register if not on auth pages themselves
    const authLinks: NavigationLink[] = [];
    if (!pathname.startsWith('/auth')) {
      authLinks.push({ name: "Login", href: "/auth/login", iconName: "LogIn" });
      authLinks.push({ name: "Register", href: "/auth/register", iconName: "UserPlus" });
    }
    
    // Filter out existing Login/Register from baseLinks if dynamicLinks already include them
    baseLinks = baseLinks.filter(link => link.href !== '/auth/login' && link.href !== '/auth/register');

    return [...baseLinks, ...authLinks];
  };

  const currentNavLinks = getEffectiveNavLinks();


  const NavLinksContent = ({ isMobile = false, closeMenu }: { isMobile?: boolean; closeMenu?: () => void }) => {
    if (isLoading && (!currentNavLinks || currentNavLinks.length === 0 || currentNavLinks.every(link => link.href.startsWith('/auth/')))) { // Adjust condition
      return (
        <>
          {[...Array(5)].map((_, i) => (
            <Skeleton key={i} className={`h-6 w-20 rounded-md ${isMobile ? "my-1" : "mx-1"}`} />
          ))}
        </>
      );
    }

    if (error && (!currentNavLinks || currentNavLinks.length === 0 || currentNavLinks.every(link => link.href.startsWith('/auth/')))) {
      return (
        <Alert variant="destructive" className="text-xs px-3 py-2 my-1 md:my-0">
            <AlertCircle className="h-3 w-3" />
            <AlertTitle className="text-xs font-normal">Menu Error</AlertTitle>
            <AlertDescription className="text-xs">Links unavailable: {error}</AlertDescription>
        </Alert>
      );
    }
    
    if ((!currentNavLinks || currentNavLinks.length === 0) && !isLoading) {
      return (
         <Alert className="text-xs px-3 py-2 my-1 md:my-0">
            <AlertCircle className="h-3 w-3" />
            <AlertTitle className="text-xs font-normal">Navigation</AlertTitle>
            <AlertDescription className="text-xs">No links available.</AlertDescription>
        </Alert>
      );
    }
    
    return (
        <>
          {currentNavLinks.map((link) => {
            if (!link || typeof link.href !== 'string' || typeof link.name !== 'string') {
              console.error("Header: Malformed navigation link object:", link);
              return null; 
            }
            const IconComponent = getIcon(link.iconName || 'LinkIconProp'); // Fallback icon

            return (
              <Link
                key={link.name}
                href={link.href}
                onClick={(e) => {
                  if (closeMenu) closeMenu();
                  if (link.href.startsWith("/#") && pathname === '/') {
                    e.preventDefault(); 
                    const sectionId = link.href.substring(2); 
                    const element = document.getElementById(sectionId);
                    if (element) {
                      element.scrollIntoView({ behavior: 'smooth' });
                    } else {
                      console.warn(`Header: Element with ID "${sectionId}" not found for scrolling.`);
                    }
                  }
                  // For other links (like /blog, /auth/login), default Link behavior will handle navigation.
                }}
                className={`px-3 py-2 rounded-md text-sm font-medium hover:bg-primary/10 hover:text-primary transition-colors ${isMobile ? "block" : ""}`}
                aria-label={`Navigate to ${link.name}`}
              >
                 <IconComponent className="mr-1.5 h-4 w-4 inline-block" aria-hidden="true" />
                {link.name}
              </Link>
            );
          })}
        </>
      );
  };


  return (
    <header className="sticky top-0 z-50 w-full border-b border-border/40 bg-background/95 backdrop-blur supports-[backdrop-filter]:bg-background/60 no-print">
      <div className="container mx-auto flex h-16 max-w-screen-2xl items-center justify-between px-4 sm:px-6 lg:px-8">
        <Link 
          href="/" 
          className="flex items-center gap-2" 
          aria-label={`Go to homepage - ${personalInfo.name.split(' ')[0]}'s resume`}
          onClick={(e) => {
            if (mobileMenuOpen) setMobileMenuOpen(false);
            if (pathname === '/') { 
              e.preventDefault(); 
              const heroElement = document.getElementById('hero'); 
              if (heroElement) {
                heroElement.scrollIntoView({ behavior: 'smooth' });
              }
            }
          }}>
          <Briefcase className="h-6 w-6 text-primary" aria-hidden="true" />
          <span className="font-bold text-lg">{personalInfo.name.split(' ')[0]}</span>
        </Link>

        <nav className="hidden md:flex items-center space-x-1" aria-label="Main navigation">
          <NavLinksContent />
        </nav>

        <div className="flex items-center gap-2">
          <ThemeToggle />
          {onDownloadResume && (
            <Button 
              onClick={onDownloadResume} 
              variant="outline" 
              size="sm" 
              className="hidden sm:flex items-center gap-2"
              aria-label="Download Resume as PDF"
            >
              <Download className="h-4 w-4" aria-hidden="true" />
              Resume
            </Button>
          )}
          <div className="md:hidden">
            <Button
              variant="ghost"
              size="icon"
              onClick={() => setMobileMenuOpen(!mobileMenuOpen)}
              aria-expanded={mobileMenuOpen}
              aria-controls="mobile-menu"
              aria-label={mobileMenuOpen ? "Close menu" : "Open menu"}
            >
              {mobileMenuOpen ? <X className="h-6 w-6" aria-hidden="true" /> : <Menu className="h-6 w-6" aria-hidden="true" />}
            </Button>
          </div>
        </div>
      </div>

      {mobileMenuOpen && (
        <motion.div
          id="mobile-menu"
          initial={{ opacity: 0, y: -20 }}
          animate={{ opacity: 1, y: 0 }}
          exit={{ opacity: 0, y: -20 }}
          className="md:hidden absolute top-16 left-0 right-0 bg-background shadow-lg p-4 z-40"
          role="navigation"
          aria-label="Mobile navigation"
        >
          <nav className="flex flex-col space-y-2">
            <NavLinksContent isMobile closeMenu={() => setMobileMenuOpen(false)} />
            {onDownloadResume && (
              <Button 
                onClick={() => { 
                  if (onDownloadResume) onDownloadResume(); 
                  setMobileMenuOpen(false); 
                }} 
                variant="outline" 
                size="sm" 
                className="w-full flex items-center gap-2"
                aria-label="Download Resume as PDF"
              >
                <Download className="h-4 w-4" aria-hidden="true" />
                Download Resume
              </Button>
            )}
          </nav>
        </motion.div>
      )}
    </header>
  );
}

