
import Link from "next/link";
import { Button } from "@/components/ui/button";
import { personalInfo, type SocialLink, iconMap, getIcon } from "@/lib/data"; // Import getIcon
import { Skeleton } from "../ui/skeleton";
import { AlertCircle } from "lucide-react";
import { Alert, AlertDescription } from "../ui/alert";


interface FooterProps {
  socialLinks: SocialLink[] | null;
  isLoading?: boolean;
  error?: string | null;
}

export function Footer({ socialLinks, isLoading, error }: FooterProps) {

  const SocialLinksContent = () => {
    if (isLoading && (!socialLinks || socialLinks.length === 0)) {
      return (
        <div className="flex justify-center space-x-4 mb-4">
          {[...Array(4)].map((_, i) => (
            <Skeleton key={i} className="h-9 w-9 rounded-full" />
          ))}
        </div>
      );
    }

    if (error && (!socialLinks || socialLinks.length === 0)) {
      return (
        <Alert variant="destructive" className="mb-2 text-xs max-w-xs mx-auto">
          <AlertCircle className="h-3 w-3" />
          <AlertDescription>Social links unavailable: {error}</AlertDescription>
        </Alert>
      );
    }

    if (!isLoading && !error && (!socialLinks || socialLinks.length === 0)) {
      return <AlertDescription className="text-xs mb-2">No social links found.</AlertDescription>;
    }

    if (socialLinks && socialLinks.length > 0) {
      return (
        <div className="flex justify-center space-x-4 mb-4" role="navigation" aria-label="Social media links">
          {socialLinks.map((link) => {
            if (!link || !link.iconName) {
              console.warn("Footer: Malformed social link or missing iconName:", link);
              return null;
            }
            const IconComponent = getIcon(link.iconName);
            return (
              <Link key={link.name} href={link.url} target="_blank" rel="noopener noreferrer" aria-label={`Visit ${personalInfo.name}'s ${link.name} profile`}>
                <Button variant="ghost" size="icon" className="text-muted-foreground hover:text-primary rounded-full">
                  {IconComponent && <IconComponent className="h-5 w-5" aria-hidden="true" />}
                </Button>
              </Link>
            );
          })}
        </div>
      );
    }
    return null;
  };


  return (
    <footer className="bg-muted/50 text-muted-foreground py-8 no-print">
      <div className="container mx-auto px-4 sm:px-6 lg:px-8 text-center">
        <SocialLinksContent />
        <p className="text-sm mb-2">
          &copy; {new Date().getFullYear()} {personalInfo.name}. All rights reserved.
        </p>
        <div className="flex flex-wrap justify-center items-center space-x-4 text-xs">
          <Link href="/terms-of-service" className="hover:text-primary hover:underline">
            Terms of Service
          </Link>
          <span className="hidden sm:inline">|</span>
          <Link href="/privacy-policy" className="hover:text-primary hover:underline">
            Privacy Policy
          </Link>
          <span className="hidden sm:inline">|</span>
          <Link href="/data-deletion" className="hover:text-primary hover:underline">
            Data Deletion
          </Link>
        </div>
      </div>
    </footer>
  );
}

