
"use client";

import { Twitter, Facebook, Linkedin, Link as LinkIcon, Copy } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { useToast } from '@/hooks/use-toast';

interface ShareButtonsProps {
  postUrl: string;
  postTitle: string;
  className?: string;
}

export function ShareButtons({ postUrl, postTitle, className }: ShareButtonsProps) {
  const { toast } = useToast();

  const encodedUrl = encodeURIComponent(postUrl);
  const encodedTitle = encodeURIComponent(postTitle);

  const shareOptions = [
    {
      name: 'Twitter',
      icon: <Twitter className="h-5 w-5" />,
      url: `https://twitter.com/intent/tweet?url=${encodedUrl}&text=${encodedTitle}`,
    },
    {
      name: 'Facebook',
      icon: <Facebook className="h-5 w-5" />,
      url: `https://www.facebook.com/sharer/sharer.php?u=${encodedUrl}`,
    },
    {
      name: 'LinkedIn',
      icon: <Linkedin className="h-5 w-5" />,
      url: `https://www.linkedin.com/shareArticle?mini=true&url=${encodedUrl}&title=${encodedTitle}`,
    },
  ];

  const handleCopyLink = async () => {
    try {
      await navigator.clipboard.writeText(postUrl);
      toast({
        title: 'Link Copied!',
        description: 'The post URL has been copied to your clipboard.',
        variant: 'default',
      });
    } catch (err) {
      console.error('Failed to copy link: ', err);
      toast({
        title: 'Error Copying Link',
        description: 'Could not copy the link to your clipboard.',
        variant: 'destructive',
      });
    }
  };

  return (
    <div className={`flex items-center gap-2 ${className}`}>
      <span className="text-sm font-medium text-muted-foreground mr-2 hidden sm:inline">Share:</span>
      {shareOptions.map((option) => (
        <Button
          key={option.name}
          variant="outline"
          size="icon"
          asChild
          className="rounded-full hover:bg-primary/10 hover:text-primary hover:border-primary"
          aria-label={`Share on ${option.name}`}
        >
          <a href={option.url} target="_blank" rel="noopener noreferrer">
            {option.icon}
          </a>
        </Button>
      ))}
      <Button
        variant="outline"
        size="icon"
        onClick={handleCopyLink}
        className="rounded-full hover:bg-primary/10 hover:text-primary hover:border-primary"
        aria-label="Copy post link"
      >
        <Copy className="h-5 w-5" />
      </Button>
    </div>
  );
}
