
"use client";

import Link from 'next/link';
import Image from 'next/image';
import type { BlogPost } from '@/lib/data';
import { Card, CardContent, CardFooter, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { ArrowRight, CalendarDays, Tag } from 'lucide-react';
import { cn } from '@/lib/utils';

interface BlogPostCardProps {
  post: BlogPost;
  viewMode?: 'grid' | 'list';
}

export function BlogPostCard({ post, viewMode = 'grid' }: BlogPostCardProps) {
  const { slug, frontmatter } = post;

  const isListView = viewMode === 'list';

  return (
    <Card 
      className={cn(
        "overflow-hidden shadow-lg hover:shadow-xl transition-shadow duration-300 bg-card text-card-foreground",
        isListView ? "flex flex-col sm:flex-row sm:items-start" : "flex flex-col h-full"
      )}
    >
      {frontmatter.image && (
        <Link 
          href={`/blog/${slug}`} 
          className={cn(
            "block relative group",
            isListView ? "sm:w-1/3 md:w-1/4 lg:w-1/5 flex-shrink-0 sm:h-auto h-48" : "w-full h-48"
          )}
        >
          <Image
            src={frontmatter.image}
            alt={frontmatter.title}
            fill
            style={{ objectFit: 'cover' }}
            sizes={isListView ? "(max-width: 640px) 100vw, 33vw" : "(max-width: 768px) 100vw, (max-width: 1200px) 50vw, 33vw"}
            className={cn(
              "transition-transform duration-300 group-hover:scale-105",
              isListView ? "sm:rounded-l-lg sm:rounded-t-none rounded-t-lg" : "rounded-t-lg"
            )}
            data-ai-hint={frontmatter.dataAiHint || 'blog post image'}
          />
        </Link>
      )}
      <div className={cn("flex flex-col flex-grow", isListView && frontmatter.image ? "sm:w-2/3 md:w-3/4 lg:w-4/5" : "")}>
        <CardHeader className="p-4">
          <CardTitle className="text-xl font-semibold text-primary hover:underline">
            <Link href={`/blog/${slug}`}>{frontmatter.title}</Link>
          </CardTitle>
          <div className="text-xs text-muted-foreground mt-1 flex items-center space-x-2 flex-wrap">
            <span className="flex items-center whitespace-nowrap"><CalendarDays className="mr-1 h-3 w-3" /> {new Date(frontmatter.date).toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' })}</span>
            <span className="flex items-center whitespace-nowrap"><Tag className="mr-1 h-3 w-3" /> {frontmatter.category}</span>
          </div>
        </CardHeader>
        <CardContent className="flex-grow p-4 pt-0">
          <CardDescription 
            className={cn(
              "text-sm text-muted-foreground min-h-[3.75rem]", // min-h to prevent layout shifts
              isListView ? "line-clamp-2" : "line-clamp-3"
            )}
          >
            {frontmatter.excerpt}
          </CardDescription>
        </CardContent>
        <CardFooter className="p-4 border-t mt-auto">
          <Link href={`/blog/${slug}`} className="text-sm font-medium text-primary hover:underline flex items-center group">
            Read More <ArrowRight className="ml-1 h-4 w-4 transition-transform group-hover:translate-x-1" />
          </Link>
        </CardFooter>
      </div>
    </Card>
  );
}
