"use client";

import { motion, type Variants } from "framer-motion";
import type { ReactNode } from "react";
import { cn } from "@/lib/utils";

interface SectionWrapperProps {
  id: string;
  title?: string;
  subtitle?: string;
  children: ReactNode;
  className?: string;
  titleClassName?: string;
  containerClassName?: string;
}

const sectionVariants: Variants = {
  hidden: { opacity: 0, filter: 'blur(5px)' },
  visible: { 
    opacity: 1, 
    filter: 'blur(0px)',
    transition: { 
      duration: 1.2, 
      ease: "easeInOut", 
      delay: 0.2 
    } 
  },
};

export function SectionWrapper({ 
  id, 
  title, 
  subtitle, 
  children, 
  className = "", 
  titleClassName = "",
  containerClassName = "container mx-auto px-4 sm:px-6 lg:px-8" 
}: SectionWrapperProps) {
  return (
    <motion.section
      id={id}
      className={cn(
        // Reduced vertical padding from py-16 md:py-24 to py-10 md:py-16
        "flex flex-col items-center justify-center py-10 md:py-16 relative overflow-hidden", 
        className 
      )}
      // scrollSnapAlign: 'start' was removed in a previous step to fix scrolling
      variants={sectionVariants}
      initial="hidden"
      whileInView="visible"
      viewport={{ once: true, amount: 0.2 }}
    >
      <div className={cn("w-full", containerClassName)}>
        {title && (
          <div className="text-center mb-12 relative">
            <h2 className={cn(
              "text-4xl md:text-5xl font-bold mb-3 relative inline-block", // Ensure h2 is inline-block for background text centering
              titleClassName
            )}>
              {/* Background Text (Decorative) */}
              <span
                aria-hidden="true"
                className="absolute inset-0 text-7xl md:text-9xl font-extrabold text-foreground opacity-[0.05] dark:opacity-[0.03] blur-sm select-none whitespace-nowrap pointer-events-none"
                style={{ zIndex: -1, top: '50%', left: '50%', transform: 'translate(-50%, -50%)' }}
              >
                {title}
              </span>
              {/* Actual Title (Animated Gradient) */}
              <span className="relative z-10 bg-clip-text text-transparent bg-gradient-to-r from-[hsl(var(--primary))] via-[hsl(var(--shimmer-highlight))] to-[hsl(var(--accent))] bg-[size:300%_300%] animate-gradient-xy">
                {title}
              </span>
            </h2>
            {subtitle && (
              <p className="text-lg md:text-xl text-muted-foreground max-w-3xl mx-auto">
                {subtitle}
              </p>
            )}
          </div>
        )}
        {children}
      </div>
    </motion.section>
  );
}
