
"use client";

import Image from "next/image";
import Link from "next/link";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription, DialogFooter, DialogClose } from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import type { Project, ProjectTech } from "@/lib/data"; 
import { ExternalLink, X } from "lucide-react";
import { ScrollArea } from "./ui/scroll-area";

interface ProjectDetailModalProps {
  project: Project;
  isOpen: boolean;
  onClose: () => void;
}

export function ProjectDetailModal({ project, isOpen, onClose }: ProjectDetailModalProps) {
  console.log(`ProjectDetailModal: Rendering. isOpen: ${isOpen}, Project title: ${project?.title}`);

  if (!isOpen || !project) {
    console.log("ProjectDetailModal: Condition not met for rendering (isOpen is false or project is null).");
    return null;
  }

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent 
        className="sm:max-w-2xl max-h-[90vh] flex flex-col bg-card text-card-foreground border-4 border-red-500" // Added red border for debugging & ensured background
        aria-labelledby="project-dialog-title"
        aria-describedby="project-dialog-description"
      >
        <DialogHeader>
          <DialogTitle id="project-dialog-title" className="text-2xl text-primary">{project.title}</DialogTitle>
          <DialogClose asChild>
            <Button 
              variant="ghost" 
              size="icon" 
              className="absolute right-4 top-4 rounded-sm opacity-70 ring-offset-background transition-opacity hover:opacity-100 focus:outline-none focus:ring-2 focus:ring-ring focus:ring-offset-2 disabled:pointer-events-none data-[state=open]:bg-accent data-[state=open]:text-muted-foreground" 
              onClick={onClose}
              aria-label="Close project details modal"
            >
              <X className="h-4 w-4" aria-hidden="true" />
            </Button>
          </DialogClose>
        </DialogHeader>
        <ScrollArea className="flex-grow pr-6 -mr-6"> 
          <div className="grid gap-4 py-4">
            <div className="relative w-full h-64 md:h-80 rounded-lg overflow-hidden">
              <Image
                src={project.image}
                alt={project.title}
                fill
                style={{ objectFit: 'cover' }}
                sizes="(max-width: 768px) 100vw, (max-width: 1200px) 50vw, 33vw"
                data-ai-hint={project.dataAiHint}
                onError={(e) => console.error(`ProjectDetailModal: Image failed to load for project "${project.title}". URL: ${project.image}`)}
              />
            </div>
            <DialogDescription id="project-dialog-description" className="text-base text-foreground/80 leading-relaxed">
              {project.description.split('\n').map((paragraph, index) => (
                <div key={index} className="mb-2 last:mb-0">{paragraph}</div>
              ))}
            </DialogDescription>
            <div>
              <h4 className="text-lg font-semibold mb-2 text-muted-foreground">Tech Stack:</h4>
              <div className="flex flex-wrap gap-2">
                {project.tech.map((techItem, techIndex) => {
                  const IconComponent = techItem.icon;
                  const isIconValidComponent = typeof IconComponent === 'function';

                  if (!isIconValidComponent && techItem.icon) {
                    console.error(`ProjectDetailModal: Invalid icon for tech "${techItem.name}" in project "${project.title}". Expected a component, got:`, IconComponent);
                  }
                  
                  return (
                    <Badge
                      key={`${techItem.name}-${techIndex}`}
                      variant="secondary"
                      className="flex items-center gap-1 text-sm bg-primary/10 text-primary hover:bg-primary/20 px-3 py-1"
                      aria-label={`Technology used: ${techItem.name}`}
                    >
                      {isIconValidComponent && <IconComponent className="h-4 w-4" aria-hidden="true"/>}
                      {techItem.name}
                    </Badge>
                  );
                })}
                 {(!project.tech || project.tech.length === 0) && (
                    <p className="text-xs text-muted-foreground">Tech stack not specified.</p>
                )}
              </div>
            </div>
          </div>
        </ScrollArea>
        <DialogFooter className="pt-4 border-t">
          {project.url && (
            <Link href={project.url} target="_blank" rel="noopener noreferrer" aria-label={`Visit project website for ${project.title}`}>
              <Button variant="outline" className="flex items-center gap-2 border-primary text-primary hover:bg-primary hover:text-primary-foreground">
                <ExternalLink className="h-4 w-4" aria-hidden="true" /> Visit Project
              </Button>
            </Link>
          )}
          <Button variant="secondary" onClick={onClose} aria-label="Close project details modal">Close</Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}
