
"use client";

import Image from "next/image";
import Link from "next/link";
import type { Project } from "@/lib/data";
import { Card, CardContent, CardFooter, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { ExternalLink, Eye } from "lucide-react";

interface ProjectCardProps {
  project: Project;
  onViewDetails: (project: Project) => void;
}

export function ProjectCard({ project, onViewDetails }: ProjectCardProps) {
  // console.log("ProjectCard: Rendering card for project -", project?.title, "Data:", project);

  if (!project || !project.title) {
    console.error("ProjectCard: Invalid project data received or title is missing.", project);
    return (
      <div className="p-4 border-2 border-orange-500 bg-orange-100 text-orange-700 rounded-lg h-full">
        Error: Invalid project data provided to ProjectCard. Title is missing.
      </div>
    );
  }

  const handleImageError = (e: React.SyntheticEvent<HTMLImageElement, Event>) => {
    console.error(`ProjectCard: Image failed to load for project "${project.title}". URL: ${project.image}. Falling back to placeholder.`);
    e.currentTarget.src = `https://placehold.co/400x300.png?text=${encodeURIComponent(project.title || "Project Error")}`;
    e.currentTarget.setAttribute('data-ai-hint', 'placeholder image error');
  };

  return (
    <Card
      className="flex flex-col overflow-hidden shadow-xl hover:shadow-2xl transition-shadow duration-300 h-full bg-card text-card-foreground dark:bg-card/80 dark:text-card-foreground/90 border-2 border-blue-500"
    >
      {project.url ? (
        <Link href={project.url} target="_blank" rel="noopener noreferrer" aria-label={`Visit ${project.title} website`}>
          <div className="relative w-full h-48 cursor-pointer group">
            <Image
              src={project.image}
              alt={project.title}
              fill
              sizes="(max-width: 768px) 100vw, (max-width: 1200px) 50vw, 33vw"
              style={{ objectFit: 'cover' }}
              className="rounded-t-lg transition-transform duration-300 group-hover:scale-105"
              data-ai-hint={project.dataAiHint || 'project image'}
              onError={handleImageError}
              priority={false} // Set to false unless it's critical for LCP
            />
            <div className="absolute inset-0 bg-black/30 group-hover:bg-black/10 transition-colors duration-300 flex items-center justify-center opacity-0 group-hover:opacity-100">
              <ExternalLink className="h-8 w-8 text-white" />
            </div>
          </div>
        </Link>
      ) : (
        <div className="relative w-full h-48">
          <Image
            src={project.image}
            alt={project.title}
            fill
            sizes="(max-width: 768px) 100vw, (max-width: 1200px) 50vw, 33vw"
            style={{ objectFit: 'cover' }}
            className="rounded-t-lg"
            data-ai-hint={project.dataAiHint || 'project image'}
            onError={handleImageError}
            priority={false}
          />
        </div>
      )}
      <CardHeader className="p-4">
        <CardTitle className="text-xl font-semibold text-primary">{project.title}</CardTitle>
        <CardDescription className="text-muted-foreground mt-1 text-sm line-clamp-3 min-h-[3.75rem]">
          {project.description}
        </CardDescription>
      </CardHeader>
      <CardContent className="flex-grow p-4">
        <div className="mb-4">
          <h4 className="text-sm font-medium mb-2 text-muted-foreground">Tech Stack:</h4>
          <div className="flex flex-wrap gap-2">
            {project.tech && project.tech.map((techItem, techIndex) => {
              if (!techItem || !techItem.name) {
                 console.error(`ProjectCard: Malformed techItem (missing name) in project "${project.title}" at index ${techIndex}:`, techItem);
                 return (
                  <Badge key={`malformed-tech-${techIndex}`} variant="destructive" className="text-xs">
                    Invalid Tech
                  </Badge>
                 );
              }

              const IconComponent = techItem.icon;
              const isIconValidComponent = typeof IconComponent === 'function';

              if (!isIconValidComponent && techItem.icon) { // Log if icon exists but is not a function
                console.error(`ProjectCard: Invalid icon for tech "${techItem.name}" in project "${project.title}". Expected a component, got:`, IconComponent);
              }

              return (
                <Badge
                  key={`${techItem.name}-${techIndex}`}
                  variant="secondary"
                  className="flex items-center gap-1 text-xs bg-primary/10 text-primary hover:bg-primary/20 px-2 py-1"
                >
                  {isIconValidComponent && <IconComponent className="h-3 w-3" />}
                  {techItem.name}
                </Badge>
              );
            })}
            {(!project.tech || project.tech.length === 0) && (
              <p className="text-xs text-muted-foreground">Tech stack not specified.</p>
            )}
          </div>
        </div>
      </CardContent>
      <CardFooter className="mt-auto p-4 border-t border-border/50">
        <div className="flex w-full justify-start space-x-2">
          <Button variant="outline" size="sm" onClick={() => onViewDetails(project)} className="flex items-center gap-2 border-primary text-primary hover:bg-primary hover:text-primary-foreground">
            <Eye className="h-4 w-4" /> View Details
          </Button>
          {project.url && (
            <Link href={project.url} target="_blank" rel="noopener noreferrer">
              <Button variant="ghost" size="sm" className="flex items-center gap-2 text-primary hover:bg-primary/10">
                <ExternalLink className="h-4 w-4" /> Visit Site
              </Button>
            </Link>
          )}
        </div>
      </CardFooter>
    </Card>
  );
}
