
import { getBlogPosts } from '@/services/apiService';
import { personalInfo } from '@/lib/data';

const APP_URL = personalInfo.siteUrl;

async function generateSitemap() {
  const today = new Date().toISOString().split('T')[0];

  let sitemap = `<?xml version="1.0" encoding="UTF-8"?>\n`;
  sitemap += `<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">\n`;

  // Static pages
  const staticPages = [
    { url: `${APP_URL}/`, changefreq: 'weekly', priority: '1.0', lastmod: today },
    { url: `${APP_URL}/blog`, changefreq: 'weekly', priority: '0.8', lastmod: today },
    { url: `${APP_URL}/auth/login`, changefreq: 'monthly', priority: '0.5', lastmod: today },
    { url: `${APP_URL}/auth/register`, changefreq: 'monthly', priority: '0.5', lastmod: today },
    { url: `${APP_URL}/data-deletion`, changefreq: 'yearly', priority: '0.3', lastmod: today },
    { url: `${APP_URL}/terms-of-service`, changefreq: 'yearly', priority: '0.3', lastmod: today },
    { url: `${APP_URL}/privacy-policy`, changefreq: 'yearly', priority: '0.3', lastmod: today },
  ];

  staticPages.forEach(page => {
    sitemap += `  <url>\n`;
    sitemap += `    <loc>${page.url}</loc>\n`;
    sitemap += `    <lastmod>${page.lastmod}</lastmod>\n`;
    sitemap += `    <changefreq>${page.changefreq}</changefreq>\n`;
    sitemap += `    <priority>${page.priority}</priority>\n`;
    sitemap += `  </url>\n`;
  });

  // Blog posts
  try {
    const postsResponse = await getBlogPosts(); // Fetches all posts
    if (postsResponse.success && postsResponse.data) {
      postsResponse.data.forEach(post => {
        if (post.slug && post.frontmatter.date) {
          const postUrl = `${APP_URL}/blog/${post.slug}`;
          const postLastMod = new Date(post.frontmatter.date).toISOString().split('T')[0];
          sitemap += `  <url>\n`;
          sitemap += `    <loc>${postUrl}</loc>\n`;
          sitemap += `    <lastmod>${postLastMod}</lastmod>\n`;
          sitemap += `    <changefreq>monthly</changefreq>\n`; // Or 'yearly' if posts are rarely updated
          sitemap += `    <priority>0.7</priority>\n`;
          sitemap += `  </url>\n`;
        }
      });
    } else {
      console.warn("Sitemap: Failed to fetch blog posts for sitemap generation.", postsResponse.error);
    }
  } catch (error) {
    console.error("Sitemap: Error fetching blog posts:", error);
  }
  
  // TODO: Add Category and Tag pages if desired in the future.
  // This would require fetching all categories and tags similar to how posts are fetched.
  // Example for categories:
  // const categories = await getBlogCategoriesFromApi();
  // categories.data?.forEach(category => { /* add to sitemap */ });
  // Example for tags:
  // const tags = await getBlogTagsFromApi();
  // tags.data?.forEach(tag => { /* add to sitemap */ });

  sitemap += `</urlset>`;
  return sitemap;
}

export async function GET() {
  const sitemap = await generateSitemap();
  return new Response(sitemap, {
    headers: {
      'Content-Type': 'application/xml',
    },
  });
}
