
"use client";

import Link from 'next/link';
import { useEffect, useState, useMemo } from 'react';
import { BlogPostCard } from '@/components/blog/BlogPostCard';
import { SectionWrapper } from '@/components/SectionWrapper';
import { Button } from '@/components/ui/button';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { LayoutGrid, List, ArrowUpDown } from 'lucide-react';
import type { BlogPost } from '@/lib/data';
import { Skeleton } from '@/components/ui/skeleton';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import { AlertCircle } from 'lucide-react';
import { getBlogPosts } from '@/services/apiService';
import { AdSensePlaceholder } from '@/components/AdSensePlaceholder'; // Import reusable component

type ViewMode = 'grid' | 'list';
type SortOption = 'newest' | 'oldest' | 'title-asc' | 'title-desc';

export default function BlogPage() {
  const [posts, setPosts] = useState<BlogPost[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [viewMode, setViewMode] = useState<ViewMode>('grid');
  const [sortOption, setSortOption] = useState<SortOption>('newest');

  useEffect(() => {
    async function fetchPosts() {
      setIsLoading(true);
      setError(null);
      try {
        const response = await getBlogPosts();
        if (response.success) {
          setPosts(response.data ?? []); 
          if (!response.data || response.data.length === 0) {
             setError(response.error || "No blog posts found.");
          } else if (response.error) { // Service might provide an informational error even on success
             setError(response.error);
          }
        } else {
          let uiErrorMessage = response.error || "Failed to fetch posts.";
          if (response.statusCode === null) { 
            uiErrorMessage = "Could not connect to the server to fetch posts. Please check your network connection.";
          } else if (response.statusCode === 404) {
            uiErrorMessage = "Could not find blog posts on the server (Error 404).";
          } else if (response.statusCode) {
            uiErrorMessage = `Failed to load blog posts (Error ${response.statusCode}).`;
          }
          setError(uiErrorMessage);
          setPosts([]);
        }
      } catch (e) {
        console.error("BlogPage: Critical error fetching posts", e);
        setError("A critical error occurred while fetching posts. Please try again later.");
        setPosts([]);
      } finally {
        setIsLoading(false);
      }
    }
    fetchPosts();
  }, []);

  const sortedAndFilteredPosts = useMemo(() => {
    let sorted = [...posts];
    const getDate = (dateString: string | undefined): number => {
        if (!dateString) return 0;
        const date = new Date(dateString);
        return isNaN(date.getTime()) ? 0 : date.getTime();
    };

    switch (sortOption) {
      case 'newest':
        sorted.sort((a, b) => getDate(b.frontmatter?.date) - getDate(a.frontmatter?.date));
        break;
      case 'oldest':
        sorted.sort((a, b) => getDate(a.frontmatter?.date) - getDate(b.frontmatter?.date));
        break;
      case 'title-asc':
        sorted.sort((a, b) => (a.frontmatter?.title || "").localeCompare(b.frontmatter?.title || ""));
        break;
      case 'title-desc':
        sorted.sort((a, b) => (b.frontmatter?.title || "").localeCompare(a.frontmatter?.title || ""));
        break;
      default:
        break;
    }
    return sorted;
  }, [posts, sortOption]);

  const BlogListSkeleton = () => (
    <div className={`gap-8 ${viewMode === 'grid' ? 'grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3' : 'flex flex-col'}`}>
      {[...Array(6)].map((_, i) => (
        <div key={i} className={`flex ${viewMode === 'grid' ? 'flex-col' : 'flex-col sm:flex-row'} space-y-3 p-4 border rounded-lg bg-card/50`}>
          <Skeleton className={`rounded-md ${viewMode === 'grid' ? 'h-[180px] w-full' : 'h-[120px] sm:h-auto sm:w-1/3 md:w-1/4 flex-shrink-0'}`} />
          <div className={`space-y-2 pt-2 ${viewMode === 'grid' ? '' : 'sm:pl-4 flex-grow'}`}>
            <Skeleton className="h-5 w-3/4 rounded" />
            <Skeleton className="h-4 w-1/2 rounded mt-1" />
            <Skeleton className="h-3 w-full rounded mt-2" />
            <Skeleton className="h-3 w-5/6 rounded mt-1" />
          </div>
        </div>
      ))}
    </div>
  );

  return (
    <SectionWrapper id="blog-listing" title="My Blog" subtitle="Sharing my thoughts and experiences.">
      <AdSensePlaceholder slot="Blog Listing - Top" />
      <div className="mb-8 flex flex-col sm:flex-row justify-between items-center gap-4">
        <div className="flex items-center gap-2">
          <Button variant={viewMode === 'grid' ? 'default' : 'outline'} size="icon" onClick={() => setViewMode('grid')} aria-label="Grid view">
            <LayoutGrid className="h-5 w-5" />
          </Button>
          <Button variant={viewMode === 'list' ? 'default' : 'outline'} size="icon" onClick={() => setViewMode('list')} aria-label="List view">
            <List className="h-5 w-5" />
          </Button>
        </div>
        <div className="w-full sm:w-auto min-w-[180px]">
          <Select value={sortOption} onValueChange={(value: SortOption) => setSortOption(value)}>
            <SelectTrigger className="w-full" aria-label="Sort posts by">
              <ArrowUpDown className="mr-2 h-4 w-4 text-muted-foreground" />
              <SelectValue placeholder="Sort by..." />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="newest">Newest First</SelectItem>
              <SelectItem value="oldest">Oldest First</SelectItem>
              <SelectItem value="title-asc">Title (A-Z)</SelectItem>
              <SelectItem value="title-desc">Title (Z-A)</SelectItem>
            </SelectContent>
          </Select>
        </div>
      </div>

      {isLoading && <BlogListSkeleton />}

      {!isLoading && error && (
        <Alert variant={(error.startsWith("No blog posts found") || error.startsWith("Could not load blog posts")) ? "default" : "destructive"} className="max-w-2xl mx-auto">
          <AlertCircle className="h-4 w-4" />
          <AlertTitle>{(error.startsWith("No blog posts found") || error.startsWith("Could not load blog posts")) ? "No Posts Yet" : "Error Loading Posts"}</AlertTitle>
          <AlertDescription>{error}</AlertDescription>
        </Alert>
      )}
      
      {!isLoading && !error && posts.length === 0 && (
        <p className="text-center text-muted-foreground text-lg py-8">No blog posts found yet. Check back soon!</p>
      )}

      {!isLoading && !error && sortedAndFilteredPosts.length > 0 && (
        <div className={`${viewMode === 'grid' ? 'grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3' : 'flex flex-col'} gap-8`}>
          {sortedAndFilteredPosts.map((post) => (
            <BlogPostCard key={post.slug} post={post} viewMode={viewMode} />
          ))}
        </div>
      )}
      <AdSensePlaceholder slot="Blog Listing - Bottom" className="mt-12" />
    </SectionWrapper>
  );
}
