
import { type NextRequest, NextResponse } from 'next/server';
import axios, { type AxiosRequestConfig, type AxiosError } from 'axios';
import { API_BASE_URL } from '@/config';

async function handler(req: NextRequest) {
  const method = req.method;

  let targetPath: string | null = null;
  let payload: any = null;
  let requestHeaders: Record<string, string> = {
    'Accept': 'application/json',
    // Forward Authorization header if present
  };

  if (req.headers.get('Authorization')) {
    requestHeaders['Authorization'] = req.headers.get('Authorization')!;
  }
   if (req.headers.get('Content-Type')) {
    requestHeaders['Content-Type'] = req.headers.get('Content-Type')!;
  }


  if (method === 'GET') {
    targetPath = req.nextUrl.searchParams.get('target');
  } else if (method === 'POST' || method === 'PUT' || method === 'PATCH' || method === 'DELETE') {
    try {
      const body = await req.json();
      targetPath = body.target;
      payload = body.payload;
    } catch (e) {
      console.error("Proxy Error: Invalid JSON body for proxy request", e);
      return NextResponse.json({ message: 'Invalid JSON body for proxy request' }, { status: 400 });
    }
  }

  if (!targetPath) {
    console.error("Proxy Error: Target path not specified for proxy.");
    return NextResponse.json({ message: 'Target path not specified for proxy' }, { status: 400 });
  }

  const targetUrl = `${API_BASE_URL}${targetPath}`;

  // Validate the constructed targetUrl before making the request
  try {
    new URL(targetUrl); // This will throw an error if targetUrl is not a valid absolute URL
  } catch (e) {
    console.error(`Proxy Error: Constructed targetUrl "${targetUrl}" is invalid. API_BASE_URL: "${API_BASE_URL}", targetPath: "${targetPath}"`, e);
    return NextResponse.json({ message: 'Proxy configuration error: Invalid target API URL constructed', error: (e as Error).message }, { status: 500 });
  }

  console.log(`Proxying request: ${method} ${targetUrl}`);

  const axiosConfig: AxiosRequestConfig = {
    method: method as any,
    url: targetUrl,
    headers: requestHeaders,
    data: payload,
    // Axios will automatically handle transforming the response to JSON if Content-Type is application/json
  };


  try {
    const apiResponse = await axios(axiosConfig);
    // Forward headers from the target API response
    const responseHeaders = new Headers();
    for (const [key, value] of Object.entries(apiResponse.headers)) {
        if (typeof value === 'string') {
            responseHeaders.set(key, value);
        } else if (Array.isArray(value)) {
            value.forEach(v => responseHeaders.append(key, v));
        }
    }
    // Ensure content-type is correctly set from the target API's response
    if (apiResponse.headers['content-type'] && !responseHeaders.has('content-type')) {
        responseHeaders.set('content-type', apiResponse.headers['content-type']);
    }


    return new Response(JSON.stringify(apiResponse.data), {
      status: apiResponse.status,
      headers: responseHeaders,
    });

  } catch (error) {
    const axiosError = error as AxiosError;
    console.error(`Error proxying to ${targetUrl}:`, axiosError.message, axiosError.code, axiosError.response?.data);
    if (axiosError.response) {
      // Forward error response from target API
      return new Response(JSON.stringify(axiosError.response.data), {
        status: axiosError.response.status,
        headers: { 'Content-Type': axiosError.response.headers['content-type'] || 'application/json' },
      });
    }
    // Handle network errors or other errors where axiosError.response is not set (e.g., the "Invalid URL" error itself if it originated from axios directly)
    return NextResponse.json({ message: 'Error proxying request', error: axiosError.message }, { status: 502 }); // Bad Gateway
  }
}

export { handler as GET, handler as POST, handler as PUT, handler as PATCH, handler as DELETE };
